###############################################################################
#   ilastik: interactive learning and segmentation toolkit
#
#       Copyright (C) 2011-2014, the ilastik developers
#                                <team@ilastik.org>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# In addition, as a special exception, the copyright holders of
# ilastik give you permission to combine ilastik with applets,
# workflows and plugins which are not covered under the GNU
# General Public License.
#
# See the LICENSE file for details. License information is also available
# on the ilastik web site at:
# 		   http://ilastik.org/license.html
###############################################################################
# Python
from abc import ABC, abstractmethod
from copy import copy, deepcopy
import collections
from collections.abc import Iterable
from functools import partial
from typing import Dict, Optional

import numpy
import vigra

# lazyflow
from lazyflow.graph import Operator, InputSlot, OutputSlot
from lazyflow.request import Request, RequestPool
from lazyflow.stype import Opaque
from lazyflow.rtype import List, SubRegion
from lazyflow.roi import roiToSlice
from lazyflow.operators.opLabelBase import OpLabelBase
from lazyflow.operators.opRelabelConsecutive import OpRelabelConsecutive
from lazyflow.operators import OpLabelVolume, OpCompressedCache, OpBlockedArrayCache
from itertools import groupby, count

import logging

logger = logging.getLogger(__name__)

from ilastik.plugins.manager import pluginManager


from ilastik.applets.base.applet import DatasetConstraintError

# These features are always calculated, but not used for prediction.
# They are needed by our gui, or by downstream applets.
default_features = {"Coord<Minimum>": {}, "Coord<Maximum>": {}, "RegionCenter": {}, "Count": {}}

# to distinguish them, they go in their own category with this name
default_features_key = "Default features"


def max_margin(d, default=(0, 0, 0)):
    """find any parameter named 'margin' in the nested feature
    dictionary 'd' and return the max.

    return 'default' if none are found.

    >>> max_margin({'plugin_one' : {'feature_one' : {'margin' : 10}}})
    [10, 10, 10]

    >>> max_margin({"p1": {"f1":{"margin":(10, 5, 2)}}})
    [10, 5, 2]

    """
    margin = default
    for features in d.values():
        for params in features.values():
            try:
                pmargin = params["margin"]
                if not isinstance(pmargin, Iterable):
                    pmargin = len(default) * [pmargin]
                margin = [max(x) for x in zip(margin, pmargin)]
            except (ValueError, KeyError):
                continue
    return margin


def make_bboxes(binary_bbox, margin):
    """Return binary label arrays for an object with margin.

    Needed to compute features in the object neighborhood

    Returns (the object + context, context only)

    """
    # object and context
    max_margin = numpy.max(margin).astype(numpy.float32)
    scaled_margin = max_margin // margin
    if len(margin) > 2:
        dt = vigra.filters.distanceTransform(
            numpy.asarray(binary_bbox, dtype=numpy.float32),
            background=True,
            pixel_pitch=numpy.asarray(scaled_margin).astype(numpy.float64),
        )
    else:
        dt = vigra.filters.distanceTransform(
            numpy.asarray(binary_bbox.squeeze(), dtype=numpy.float32),
            pixel_pitch=numpy.asarray(scaled_margin).astype(numpy.float64),
        )
        dt = dt.reshape(dt.shape + (1,))

    assert dt.ndim == 3
    passed = numpy.asarray(dt < max_margin).astype(bool)

    # context only
    context = numpy.asarray(passed) ^ numpy.asarray(binary_bbox).astype(bool)
    return passed, context


class OpCachedRegionFeatures(Operator):
    """Caches the region features computed by OpRegionFeatures."""

    RawImage = InputSlot()
    Atlas = InputSlot(optional=True)
    ObjectIDMapping = InputSlot(optional=True)
    LabelImage = InputSlot()
    CacheInput = InputSlot(optional=True)
    Features = InputSlot(rtype=List, stype=Opaque)

    Output = OutputSlot()
    CleanBlocks = OutputSlot()

    # Schematic:
    #
    # RawImage -----   blockshape=(t,)=(1,)
    #               \                        \
    # LabelImage ----> OpRegionFeatures ----> OpBlockedArrayCache --> Output
    #                                                            \
    #                                                             --> CleanBlocks

    def __init__(self, *args, **kwargs):
        super(OpCachedRegionFeatures, self).__init__(*args, **kwargs)

        # Hook up the actual region features operator
        self._opRegionFeatures = OpRegionFeatures(parent=self)
        self._opRegionFeatures.RawVolume.connect(self.RawImage)
        self._opRegionFeatures.Atlas.connect(self.Atlas)
        self._opRegionFeatures.ObjectIDMapping.connect(self.ObjectIDMapping)
        self._opRegionFeatures.LabelVolume.connect(self.LabelImage)
        self._opRegionFeatures.Features.connect(self.Features)

        # Hook up the cache.
        self._opCache = OpBlockedArrayCache(parent=self)
        self._opCache.name = "OpCachedRegionFeatures._opCache"
        self._opCache.Input.connect(self._opRegionFeatures.Output)

        # Hook up our output slots
        self.Output.connect(self._opCache.Output)
        self.CleanBlocks.connect(self._opCache.CleanBlocks)

    def setupOutputs(self):
        assert self.LabelImage.meta.axistags == self.RawImage.meta.axistags

        taggedOutputShape = self.LabelImage.meta.getTaggedShape()
        taggedRawShape = self.RawImage.meta.getTaggedShape()

        if not numpy.all(list(taggedOutputShape.get(k, 0) == taggedRawShape.get(k, 0) for k in "txyz")):
            raise DatasetConstraintError(
                "Object Extraction",
                "Raw Image and Label Image shapes do not match.\n"
                "Label Image shape: {}. Raw Image shape: {}"
                "".format(self.LabelImage.meta.shape, self.RawVolume.meta.shape),
            )

        # Every value in the regionfeatures output is cached seperately as it's own "block"
        blockshape = (1,) * len(self._opRegionFeatures.Output.meta.shape)
        self._opCache.BlockShape.setValue(blockshape)

    def _setInSlot(self, slot, subindex, roi, value):
        assert slot == self.CacheInput
        slicing = roiToSlice(roi.start, roi.stop)
        self._opCache.Input[slicing] = value

    def execute(self, slot, subindex, roi, destination):
        assert False, "Shouldn't get here."

    def propagateDirty(self, slot, subindex, roi):
        pass  # Nothing to do...


class OpAdaptTimeListRoi(Operator):
    """Adapts the t array output from OpRegionFeatures to an Output
    slot that is called with a 'List' rtype, where the roi is a list
    of time slices, and the output is a dictionary of (time,
    featuredict) pairs.

    """

    Input = InputSlot()
    Output = OutputSlot(stype=Opaque, rtype=List)

    def setupOutputs(self):
        # Number of time steps
        self.Output.meta.shape = (self.Input.meta.getTaggedShape()["t"],)
        self.Output.meta.dtype = object

    def execute(self, slot, subindex, roi, destination):
        assert slot == self.Output, "Unknown output slot"
        taggedShape = self.Input.meta.getTaggedShape()

        # Special case: An empty roi list means "request everything"
        if len(roi) == 0:
            roi = list(range(taggedShape["t"]))

        taggedShape["t"] = 1
        timeIndex = list(taggedShape.keys()).index("t")

        # Get time ranges with consecutive numbers
        time_ranges = [list(g) for _, g in groupby(roi, key=lambda n, c=count(): n - next(c))]

        result = {}
        for time_range in time_ranges:
            start = [time_range[0]]
            stop = [time_range[-1] + 1]

            val = self.Input(start, stop).wait()

            for i, t in enumerate(time_range):
                result[t] = val[i]

        return result

    def propagateDirty(self, slot, subindex, roi):
        assert slot == self.Input
        timeIndex = self.Input.meta.axistags.index("t")
        self.Output.setDirty(List(self.Output, list(range(roi.start[timeIndex], roi.stop[timeIndex]))))


class OpObjectCenterImage(Operator):
    """Produceds an image with a cross in the center of each connected
    component.

    """

    SegmentationImage = InputSlot()
    RegionCenters = InputSlot(rtype=List, stype=Opaque)
    Output = OutputSlot()

    def setupOutputs(self):
        self.Output.meta.assignFrom(self.SegmentationImage.meta)

    @staticmethod
    def __contained_in_subregion(roi, coords):
        b = True
        for i in range(len(coords)):
            b = b and (roi.start[i] <= coords[i] and coords[i] < roi.stop[i])
        return b

    @staticmethod
    def __make_key(roi, coords):
        key = [int(coords[i] - roi.start[i]) for i in range(len(roi.start))]
        return tuple(key)

    def execute(self, slot, subindex, roi, result):
        assert slot == self.Output, "Unknown output slot"

        result[:] = 0
        ndim = 3
        taggedShape = self.SegmentationImage.meta.getTaggedShape()
        if "z" not in taggedShape or taggedShape["z"] == 1:
            ndim = 2
        for t in range(roi.start[0], roi.stop[0]):
            obj_features = self.RegionCenters([t]).wait()
            # FIXME: this assumes that channel is the last axis
            for ch in range(roi.start[-1], roi.stop[-1]):
                centers = obj_features[t][default_features_key]["RegionCenter"]
                if centers.size:
                    centers = centers[1:, :]
                for center in centers:
                    if ndim == 3:
                        x, y, z = center[0:3]
                        c = (t, x, y, z, ch)
                    elif ndim == 2:
                        x, y = center[0:2]
                        c = (t, x, y, 0, ch)
                    if self.__contained_in_subregion(roi, c):
                        result[self.__make_key(roi, c)] = 1

        return result

    def propagateDirty(self, slot, subindex, roi):
        # the roi here is a list of time steps
        if slot is self.RegionCenters:
            roi = list(roi)
            t = roi[0]
            T = t + 1
            a = t
            for b in roi[1:]:
                assert b - 1 == a, "List roi must be contiguous"
                a = b
                T += 1
            time_index = self.SegmentationImage.meta.axistags.index("t")
            stop = numpy.asarray(self.SegmentationImage.meta.shape, dtype=numpy.int64)
            start = numpy.zeros_like(stop)
            stop[time_index] = T
            start[time_index] = t

            roi = SubRegion(self.Output, tuple(start), tuple(stop))
            self.Output.setDirty(roi)


class OpObjectExtractionBase(Operator, ABC):
    RawImage = InputSlot()
    SegmentationImage = InputSlot()  # binary or label image

    Atlas = InputSlot(optional=True)
    BackgroundLabels = InputSlot(optional=True)

    # Bypass cache (for headless mode)
    BypassModeEnabled = InputSlot(value=False)

    # which features to compute.
    # nested dictionary with format:
    # dict[plugin_name][feature_name][parameter_name] = parameter_value
    # for example {"Standard Object Features": {"Mean in neighborhood":{"margin": (5, 5, 2)}}}
    Features = InputSlot(rtype=List, stype=Opaque, value={})

    LabelImage = OutputSlot()
    ObjectCenterImage = OutputSlot()
    RelabelDict = OutputSlot()

    # the computed features.
    # dict[time_slice][plugin_name][feature_name] = feature_value
    # by requesting from this slot with indices, the corresponding indices will be added to the dict
    # -> RegionFeatures[0].wait() -> {0: {...}}
    RegionFeatures = OutputSlot(stype=Opaque, rtype=List)

    BlockwiseRegionFeatures = OutputSlot()  # For compatibility with tracking workflow, the RegionFeatures output
    # has rtype=List, indexed by t.
    # For other workflows, output has rtype=ArrayLike, indexed by (t)

    CleanLabelBlocks = OutputSlot()
    LabelImageCacheInput = InputSlot(optional=True)
    RelabelCacheInput = InputSlot(optional=True)
    RelabelCacheOutput = OutputSlot()

    RegionFeaturesCacheInput = InputSlot(optional=True)
    RegionFeaturesCleanBlocks = OutputSlot()

    # Schematic:
    #
    # BackgroundLabels               LabelImage
    #                 \             /
    # SegmentationImage ---> EnsureConsecutiveLabels ---> opRegFeats ---> opRegFeatsAdaptOutput ---> RegionFeatures
    #                                   /                                     \
    # RawImage--------------------------                      LabelImage ---> opObjectCenterImage --> opCenterCache --> ObjectCenterImage
    #
    # EnsureConsecutiveLabels should be customized via `_create_label_volume_op`,
    # enabling different segmentation input types: binary, and label image.

    def __init__(self, parent=None, graph=None):

        super().__init__(parent, graph)

        self._opLabelVolume = self._create_label_volume_op()

        self._opRegFeats = OpCachedRegionFeatures(parent=self)
        self._opRegFeatsAdaptOutput = OpAdaptTimeListRoi(parent=self)
        self._opObjectCenterImage = OpObjectCenterImage(parent=self)
        self._opRegFeats.RawImage.connect(self.RawImage)
        self._opRegFeats.LabelImage.connect(self._opLabelVolume.CachedOutput)
        self._opRegFeats.Features.connect(self.Features)
        self._opRegFeats.Atlas.connect(self.Atlas)  # move into constructor?
        self.RegionFeaturesCleanBlocks.connect(self._opRegFeats.CleanBlocks)

        self._opRegFeats.CacheInput.connect(self.RegionFeaturesCacheInput)

        self._opRegFeatsAdaptOutput.Input.connect(self._opRegFeats.Output)

        self._opObjectCenterImage.SegmentationImage.connect(self._opLabelVolume.CachedOutput)
        self._opObjectCenterImage.RegionCenters.connect(self._opRegFeatsAdaptOutput.Output)

        self._opCenterCache = OpCompressedCache(parent=self)
        self._opCenterCache.name = "OpObjectExtraction._opCenterCache"
        self._opCenterCache.Input.connect(self._opObjectCenterImage.Output)

        # connect outputs
        self.LabelImage.connect(self._opLabelVolume.CachedOutput)
        self.ObjectCenterImage.connect(self._opCenterCache.Output)
        self.RegionFeatures.connect(self._opRegFeatsAdaptOutput.Output)
        self.BlockwiseRegionFeatures.connect(self._opRegFeats.Output)
        self.CleanLabelBlocks.connect(self._opLabelVolume.CleanBlocks)

    def setupOutputs(self):
        self._checkConstraints()

        taggedShape = self.RawImage.meta.getTaggedShape()
        for k in list(taggedShape.keys()):
            if k == "t" or k == "c":
                taggedShape[k] = 1
        self._opCenterCache.BlockShape.setValue(tuple(taggedShape.values()))

    def _checkConstraints(self, *_):
        if not any((self.RawImage.ready(), self.SegmentationImage.ready(), self.Atlas.ready())):
            return

        input_pairs = [
            (self.RawImage, self.SegmentationImage, "Raw data", "label data"),
            (self.RawImage, self.Atlas, "Raw data", "atlas data"),
            (self.SegmentationImage, self.Atlas, "Label data", "atlas data"),
        ]

        for slot1, slot2, name1, name2 in input_pairs:
            if not (slot1.ready() and slot2.ready()):
                continue
            shape1 = slot1.meta.getTaggedShape()
            shape2 = slot2.meta.getTaggedShape()
            shape1["c"] = None
            shape2["c"] = None
            if dict(shape1) != dict(shape2):
                msg = (
                    f"{name1} and {name2} must have equal dimensions (different channels are okay). "
                    f"Your datasets have shapes: {slot1.meta.shape} and {slot2.meta.shape}."
                )
                logger.info(msg)
                raise DatasetConstraintError("Object Extraction", msg)

    def _setInSlot(self, slot, subindex, roi, value):
        assert slot in [
            self.RegionFeaturesCacheInput,
            self.LabelImageCacheInput,
            self.RelabelCacheInput,
        ], "Invalid slot for _setInSlot(): {}".format(slot.name)

    def execute(self, slot, subindex, roi, result):
        assert False, "Shouldn't get here."

    def propagateDirty(self, inputSlot, subindex, roi):
        pass

    @abstractmethod
    def _create_label_volume_op(self) -> OpLabelBase:
        """Setup operator that provides label image slots

        Returns:
            label_operator: Operator with
              CachedOutput
              CleanLabelBlocks

        """
        ...


class OpObjectExtractionFromLabels(OpObjectExtractionBase):

    def __init__(self, parent=None, graph=None):
        super().__init__(parent, graph)
        self._opLabelVolume: OpRelabelConsecutive
        self._opRegFeats.ObjectIDMapping.connect(self._opLabelVolume.RelabelDict)
        self._opLabelVolume.SerializationInput.connect(self.RelabelCacheInput)
        self.RelabelCacheOutput.connect(self._opLabelVolume.SerializationOutput)
        self.RelabelDict.connect(self._opLabelVolume.RelabelDict)

    def _create_label_volume_op(self):
        opLabelVolume = OpRelabelConsecutive(parent=self)
        opLabelVolume.name = "OpObjectExtraction._opLabelVolume[OpRelabelConsecutive]"

        opLabelVolume.Input.connect(self.SegmentationImage)

        return opLabelVolume


class OpObjectExtraction(OpObjectExtractionBase):
    """The top-level operator for the object extraction applet.

    Computes object features and object center images.
    """

    def _create_label_volume_op(self) -> OpLabelBase:
        opLabelVolume = OpLabelVolume(parent=self)
        opLabelVolume.name = "OpObjectExtraction._opLabelVolume"

        opLabelVolume.Input.connect(self.SegmentationImage)
        opLabelVolume.Background.connect(self.BackgroundLabels)
        # TODO: investigate if Bypassing the cache in headless is a good idea at all!
        opLabelVolume.BypassModeEnabled.connect(self.BypassModeEnabled)
        opLabelVolume.SerializationInput.connect(self.LabelImageCacheInput)

        return opLabelVolume


class OpRegionFeatures(Operator):
    """Produces region features for time-stacked 3d+c volumes

    The image's axes are extended to the full xyzc shape, one t-slice at a
    time.

    Inputs:

    * RawVolume : the raw data on which to compute features

    * LabelVolume : a volume of connected components for each object
      in the raw data.

    * Features : a nested dictionary of features to compute.
      Features[plugin name][feature name][parameter name] = parameter value

    Outputs:

    * Output : a nested dictionary of features.
      Output[plugin name][feature name] = numpy.ndarray

    """

    RawVolume = InputSlot()
    Atlas = InputSlot(optional=True)
    ObjectIDMapping = InputSlot(optional=True)
    LabelVolume = InputSlot()
    Features = InputSlot(rtype=List, stype=Opaque)

    Output = OutputSlot()

    def setupOutputs(self):
        if self.LabelVolume.meta.axistags != self.RawVolume.meta.axistags:
            raise DatasetConstraintError(
                "Object Feature Selection",
                f"Raw and label axis tags do not match ({self.RawVolume.meta.axistags} != {self.LabelVolume.meta.axistags}).",
            )

        taggedOutputShape = self.LabelVolume.meta.getTaggedShape()
        taggedRawShape = self.RawVolume.meta.getTaggedShape()

        if not numpy.all(list(taggedOutputShape.get(k, 0) == taggedRawShape.get(k, 0) for k in "txyz")):
            raise DatasetConstraintError(
                "Object Feature Selection",
                f"Shapes do not match. Label volume shape: {self.LabelVolume.meta.shape} Raw data shape: {self.RawVolume.meta.shape}.",
            )

        self.Output.meta.shape = (taggedOutputShape["t"],)
        self.Output.meta.axistags = vigra.defaultAxistags("t")
        # The features for the entire block (in xyz) are provided for the requested tc coordinates.
        self.Output.meta.dtype = object

    def execute(self, slot, subindex, roi, result):
        assert len(roi.start) == len(roi.stop) == len(self.Output.meta.shape)
        assert slot == self.Output

        t_ind = self.RawVolume.meta.axistags.index("t")
        assert t_ind < len(self.RawVolume.meta.shape)

        def compute_features_for_time_slice(res_t_ind, t):
            axes4d = [k for k in self.RawVolume.meta.getTaggedShape().keys() if k in "xyzc"]

            # Process entire spatial volume
            s = [slice(None)] * len(self.RawVolume.meta.shape)
            s[t_ind] = slice(t, t + 1)
            s = tuple(s)

            # Request in parallel
            raw_req = self.RawVolume[s]
            raw_req.submit()

            label_req = self.LabelVolume[s]
            label_req.submit()

            if self.Atlas.ready():
                atlasVolume = self.Atlas[s].wait()
                atlasVolume = vigra.taggedView(atlasVolume, axistags=self.Atlas.meta.axistags)
                atlasVolume = atlasVolume.withAxes(*axes4d)
            else:
                atlasVolume = None

            if self.ObjectIDMapping.ready():
                object_id_mapping = self.ObjectIDMapping[s[t_ind]].wait()[0]
            else:
                object_id_mapping = None

            # Get results
            rawVolume = raw_req.wait()
            labelVolume = label_req.wait()

            rawVolume = vigra.taggedView(rawVolume, axistags=self.RawVolume.meta.axistags)
            labelVolume = vigra.taggedView(labelVolume, axistags=self.LabelVolume.meta.axistags)

            # Convert to 4D (preserve axis order)
            rawVolume = rawVolume.withAxes(*axes4d)
            labelVolume = labelVolume.withAxes(*axes4d)
            acc = self._extract(rawVolume, labelVolume, atlasVolume, object_id_mapping)

            # Copy into the result
            result[res_t_ind] = acc

        # loop over requested time slices
        pool = RequestPool()
        for res_t_ind, t in enumerate(range(roi.start[t_ind], roi.stop[t_ind])):
            pool.add(Request(partial(compute_features_for_time_slice, res_t_ind, t)))

        pool.wait()
        return result

    def compute_extent(self, i, image, mincoords, maxcoords, axes, margin):
        """Make a slicing to extract object i from the image."""
        # find the bounding box (margin is always 'xyz' order)
        result = [None] * 3
        minx = max(mincoords[i][axes.x] - margin[axes.x], 0)
        miny = max(mincoords[i][axes.y] - margin[axes.y], 0)

        # Coord<Minimum> and Coord<Maximum> give us the [min,max]
        # coords of the object, but we want the bounding box: [min,max), so add 1
        maxx = min(maxcoords[i][axes.x] + 1 + margin[axes.x], image.shape[axes.x])
        maxy = min(maxcoords[i][axes.y] + 1 + margin[axes.y], image.shape[axes.y])

        result[axes.x] = slice(minx, maxx)
        result[axes.y] = slice(miny, maxy)

        try:
            minz = max(mincoords[i][axes.z] - margin[axes.z], 0)
            maxz = min(maxcoords[i][axes.z] + 1 + margin[axes.z], image.shape[axes.z])
        except:
            minz = 0
            maxz = 1

        result[axes.z] = slice(minz, maxz)

        return result

    def compute_rawbbox(self, image, extent, axes):
        """essentially returns image[extent], preserving all channels."""
        key = copy(extent)
        key.insert(axes.c, slice(None))
        return image[tuple(key)]

    def _augmentFeatureNames(self, features):
        # Take a dictionary of feature names, augment it by default features and set to Features() slot

        feature_names = features
        feature_names_with_default = deepcopy(feature_names)

        # expand the feature list by our default features
        logger.debug("attaching default features {} to vigra features {}".format(default_features, feature_names))
        plugin = pluginManager.getPluginByName("Standard Object Features", "ObjectFeatures")
        all_default_props = plugin.plugin_object.fill_properties(default_features)  # fill in display name and such
        feature_names_with_default[default_features_key] = all_default_props

        if not "Standard Object Features" in list(feature_names.keys()):
            # The user has not selected any standard features. Add them now
            feature_names_with_default["Standard Object Features"] = {}

        for default_feature_name, default_feature_props in default_features.items():
            if default_feature_name not in feature_names_with_default["Standard Object Features"]:
                # this feature has not been selected by the user, add it now.
                feature_names_with_default["Standard Object Features"][default_feature_name] = all_default_props[
                    default_feature_name
                ]
                feature_names_with_default["Standard Object Features"][default_feature_name]["selected"] = False

        return feature_names_with_default

    def _createAtlasMapping(self, region_centers: numpy.ndarray, atlasImage: numpy.ndarray):
        axes = {tag.key: idx for idx, tag in enumerate(atlasImage.axistags)}
        num_center_points = len(region_centers)
        num_channels_in_atlas = atlasImage.shape[axes["c"]] if "c" in axes else 1
        atlas_mapping = numpy.zeros(num_center_points * num_channels_in_atlas, dtype=atlasImage.dtype)
        atlas_mapping = atlas_mapping.reshape(num_center_points, num_channels_in_atlas)
        for obj_idx, center_coords in enumerate(region_centers):
            rounded_coords = center_coords.round().astype(numpy.uint64)
            slicing = [0] * len(axes)
            slicing[axes["c"]] = slice(None)

            slicing[axes["x"]] = rounded_coords[0]
            slicing[axes["y"]] = rounded_coords[1]
            if len(center_coords) == 3:
                slicing[axes["z"]] = rounded_coords[2]

            atlas_value = atlasImage[slicing]
            atlas_mapping[obj_idx] = atlas_value
        return atlas_mapping

    def _extract(
        self, image, labels, atlas=None, object_id_mapping: Optional[dict[int, int]] = None
    ) -> Dict[str, Dict[str, numpy.ndarray]]:
        """
        Returns a dictionary of features, with the following structure:
            dict[plugin_name][feature_name] = feature_value

            where feature value is 2D numpy array with feature values per object
        """

        if not (image.ndim == labels.ndim == 4):
            raise Exception(
                "both images must be 4D. raw image shape: {} label image shape: {}".format(image.shape, labels.shape)
            )

        # FIXME: maybe simplify? taggedShape should be easier here
        class Axes(object):
            x = image.axistags.index("x")
            y = image.axistags.index("y")
            z = image.axistags.index("z")
            c = image.axistags.index("c")

        axes = Axes()

        slc3d = [slice(None)] * 4  # FIXME: do not hardcode
        slc3d[axes.c] = 0
        slc3d = tuple(slc3d)

        labels = labels[slc3d]

        # These are the feature names, selected by the user and the default feature names.
        feature_names = deepcopy(self.Features([]).wait())
        feature_names = self._augmentFeatureNames(feature_names)

        # do global features
        logger.debug("Computing global and default features")
        global_features = {}
        pool = RequestPool()

        def compute_for_one_plugin(plugin_name, feature_dict):
            plugin_inner = pluginManager.getPluginByName(plugin_name, "ObjectFeatures")
            assert plugin_inner, f"object features plugin {plugin_name} missing!"
            global_features[plugin_name] = plugin_inner.plugin_object.compute_global(image, labels, feature_dict, axes)

        for plugin_name, feature_dict in feature_names.items():
            if plugin_name != default_features_key:
                pool.add(Request(partial(compute_for_one_plugin, plugin_name, feature_dict)))

        pool.wait()

        extrafeats = {}
        for feat_key in default_features:
            try:
                sel = feature_names["Standard Object Features"][feat_key]["selected"]
            except KeyError:
                # we don't always set this property to True, sometimes it's just not there. The only important
                # thing is that it's not False
                sel = True
            if not sel:
                # This feature has not been selected by the user. Remove it from the computed dict into a special dict
                # for default features
                feature = global_features["Standard Object Features"].pop(feat_key)
            else:
                feature = global_features["Standard Object Features"][feat_key]
            extrafeats[feat_key] = feature

        if atlas is not None:
            extrafeats["AtlasMapping"] = self._createAtlasMapping(extrafeats["RegionCenter"], atlas)

        if object_id_mapping is not None:
            rev_mapping = {v: k for k, v in object_id_mapping.items()}
            extrafeats["original_oid"] = numpy.expand_dims(
                numpy.vectorize(rev_mapping.get)(numpy.arange(1, extrafeats["Count"].shape[0] + 1)), axis=-1
            )

        extrafeats = dict((k.replace(" ", ""), v) for k, v in extrafeats.items())

        # index in those have an -1 offset to object ids
        mincoords = extrafeats["Coord<Minimum>"].astype(int)
        maxcoords = extrafeats["Coord<Maximum>"].astype(int)
        nobj = mincoords.shape[0]

        local_features = collections.defaultdict(lambda: collections.defaultdict(list))
        margin = max_margin(feature_names)

        if numpy.any(margin):
            bboxes = {}
            for plugin_name, feature_dict in feature_names.items():
                if not any("margin" in features for features in feature_dict.values()):
                    continue

                plugin = pluginManager.getPluginByName(plugin_name, "ObjectFeatures")
                tmp_dicts = [None] * nobj

                def _calc_single(i, raw_bbox, binary_bbox):
                    feats = plugin.plugin_object.compute_local(raw_bbox, binary_bbox, feature_dict, axes)
                    tmp_dicts[i] = feats

                with RequestPool() as pool:
                    # starting from 0, we stripped 0th background object in global computation
                    for i in range(nobj):
                        logger.debug("processing object {}".format(i))
                        if i not in bboxes:
                            extent = self.compute_extent(i, image, mincoords, maxcoords, axes, margin)
                            raw_bbox = self.compute_rawbbox(image, extent, axes)
                            # it's i+1 here, because the background has label 0
                            binary_bbox = labels[tuple(extent)] == i + 1
                            bboxes[i] = (raw_bbox, binary_bbox)

                        raw_bbox, binary_bbox = bboxes[i]
                        pool.add(Request(partial(_calc_single, i, raw_bbox, binary_bbox)))

                # merge the results
                for feature_dict in tmp_dicts:
                    for feature_name, features in feature_dict.items():
                        local_features[plugin_name][feature_name].append(features)

        logger.debug("computing done, removing failures")
        # remove local features that failed
        for pname, pfeats in local_features.items():
            for key in list(pfeats.keys()):
                value = pfeats[key]
                try:
                    pfeats[key] = numpy.vstack(list(v.reshape(1, -1) for v in value))
                except:
                    logger.warning("feature {} failed".format(key))
                    del pfeats[key]

        # merge the global and local features
        logger.debug("removed failed, merging")
        all_features = {}
        plugin_names = set(global_features.keys()) | set(local_features.keys())
        for name in plugin_names:
            d1 = global_features.get(name, {})
            d2 = local_features.get(name, {})
            all_features[name] = dict(list(d1.items()) + list(d2.items()))
        all_features[default_features_key] = extrafeats

        # reshape all features
        for plugin_name, pfeats in all_features.items():
            for key, value in pfeats.items():
                if value.shape[0] != nobj:
                    raise Exception(
                        "feature {} does not have enough rows, {} instead of {}".format(key, value.shape[0], nobj)
                    )

                # because object classification operator expects nobj to
                # include background. FIXME: we should change that assumption.
                value = numpy.vstack((numpy.zeros(value.shape[1], dtype=value.dtype), value))

                #
                if plugin_name != default_features_key and key not in ["original_oid", "AtlasMapping"]:
                    value = value.astype(numpy.float32)  # turn Nones into numpy.NaNs

                assert value.shape[0] == nobj + 1
                assert value.ndim == 2

                pfeats[key] = value
        logger.debug("merged, returning")
        return all_features

    def propagateDirty(self, slot, subindex, roi):
        if slot is self.Features:
            self.Output.setDirty(slice(None))
        else:
            axes = list(self.RawVolume.meta.getTaggedShape().keys())
            dirtyStart = collections.OrderedDict(list(zip(axes, roi.start)))
            dirtyStop = collections.OrderedDict(list(zip(axes, roi.stop)))

            # Remove the spatial and channel dims (keep t, if present)
            del dirtyStart["x"]
            del dirtyStart["y"]
            del dirtyStart["z"]
            del dirtyStart["c"]

            del dirtyStop["x"]
            del dirtyStop["y"]
            del dirtyStop["z"]
            del dirtyStop["c"]

            self.Output.setDirty(list(dirtyStart.values()), list(dirtyStop.values()))
