###############################################################################
#   ilastik: interactive learning and segmentation toolkit
#
#       Copyright (C) 2011-2024, the ilastik developers
#                                <team@ilastik.org>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# In addition, as a special exception, the copyright holders of
# ilastik give you permission to combine ilastik with applets,
# workflows and plugins which are not covered under the GNU
# General Public License.
#
# See the LICENSE file for details. License information is also available
# on the ilastik web site at:
# 		   http://ilastik.org/license.html
###############################################################################
import os
from functools import partial
from typing import Union

from qtpy import uic
from qtpy.QtCore import Qt, Signal
from qtpy.QtGui import QColor
from qtpy.QtWidgets import QColorDialog, QDialog, QMenu, QMessageBox, QPushButton, QVBoxLayout, QWidget

from .labelListModel import Label, LabelListModel
from .listView import ListView


class ColorDialog(QDialog):
    def __init__(self, brushcolor: QColor, pmapcolor: QColor, parent=None):
        QDialog.__init__(self, parent)
        self._brushColor: QColor = brushcolor
        self._pmapColor: QColor = pmapcolor
        self.ui = uic.loadUi(os.path.join(os.path.split(__file__)[0], "color_dialog.ui"), self)

        self.ui.brushColorButton.clicked.connect(self.onBrushColor)
        self.ui.pmapColorButton.clicked.connect(self.onPmapColor)
        self.ui.linkColorCheckBox.setChecked(self._brushColor == self._pmapColor)
        self.ui.linkColorCheckBox.toggled.connect(self.onLinkColorButton)

        # initializes all button / indicator states:
        self.update_state()

    def onLinkColorButton(self, checked: bool):
        if checked:
            self._pmapColor = self._brushColor

        self.update_state()

    def update_state(self):
        self.ui.brushColorButton.setStyleSheet(f"background-color: {self._brushColor.name()}")
        self.ui.pmapColorButton.setStyleSheet(f"background-color: {self._pmapColor.name()}")

    def _getColor(self, initial: QColor) -> Union[QColor, None]:
        dlg = QColorDialog(initial, self)
        res = dlg.exec() != 0
        return dlg.selectedColor() if res else None

    def onBrushColor(self):
        color = self._getColor(self._brushColor)

        if color is None:
            return

        self._brushColor = color

        if self.ui.linkColorCheckBox.isChecked():
            self._pmapColor = color

        self.update_state()

    def brushColor(self):
        return self._brushColor

    def onPmapColor(self):
        color = self._getColor(self._pmapColor)

        if color is None:
            return

        self._pmapColor = color
        if self.ui.linkColorCheckBox.isChecked():
            self._brushColor = color

        self.update_state()

    def pmapColor(self):
        return self._pmapColor


class LabelListView(ListView):
    clearRequested = Signal(int, str)  # row, name
    mergeRequested = Signal(int, str, int, str)  # from_row, from_name, to_row, to_name

    def __init__(self, parent=None):
        super(LabelListView, self).__init__(parent=parent)
        self.support_merges = False
        self.resetEmptyMessage("no labels defined yet")

    def tableViewCellDoubleClicked(self, modelIndex):
        if modelIndex.column() == self.model.ColumnID.Color:
            colorDialog = ColorDialog(
                brushcolor=self._table.model()[modelIndex.row()].brushColor(),
                pmapcolor=self._table.model()[modelIndex.row()].pmapColor(),
                parent=self,
            )
            res = colorDialog.exec_()
            if res:
                self.model.setData(modelIndex, (colorDialog.brushColor(), colorDialog.pmapColor()))

    def tableViewCellClicked(self, modelIndex):
        if modelIndex.row() in self.model.unremovable_rows:
            return
        if modelIndex.column() == self.model.ColumnID.Delete and not self.model.flags(modelIndex) == Qt.NoItemFlags:
            current_row = modelIndex.row()
            model = modelIndex.model()
            label_name = model.data(model.index(current_row, 1), Qt.DisplayRole)
            message = (
                f"You are about to delete label '{label_name}' along with all associated annotations!\nAre you sure?"
            )
            buttons = QMessageBox.Ok | QMessageBox.Cancel
            response = QMessageBox.warning(self, "Delete Label?", message, buttons)
            if response == QMessageBox.Cancel:
                return
            elif response == QMessageBox.Ok:
                self.model.removeRow(modelIndex.row())

    def contextMenuEvent(self, event):
        from_index = self._table.indexAt(event.pos())
        if not (0 <= from_index.row() < self.model.rowCount()):
            return

        # The context menu event is the same regardless of the selected column
        # Therefore ColumnID is set such that the label name is retrieved
        from_index_to_name = self.model.index(from_index.row(), LabelListModel.ColumnID.Name)

        from_name = self.model.data(from_index_to_name, Qt.DisplayRole)
        menu = QMenu(parent=self)
        menu.addAction(
            "Clear {}".format(from_name), partial(self.clearRequested.emit, from_index_to_name.row(), str(from_name))
        )

        if self.support_merges and self.allowDelete:
            for to_row in range(self.model.rowCount()):
                to_index = self.model.index(to_row, LabelListModel.ColumnID.Name)
                to_name = self.model.data(to_index, Qt.DisplayRole)
                action = menu.addAction(
                    "Merge {} into {}".format(from_name, to_name),
                    partial(self.mergeRequested.emit, from_index_to_name.row(), str(from_name), to_row, str(to_name)),
                )
                if to_row == from_index_to_name.row():
                    action.setEnabled(False)

        menu.exec_(self.mapToGlobal(event.pos()))


if __name__ == "__main__":
    import sys

    import numpy
    from qtpy.QtWidgets import QApplication

    app = QApplication(sys.argv)

    red = QColor(255, 0, 0)
    green = QColor(0, 255, 0)
    blue = QColor(0, 0, 255)
    # model = LabelListModel([Label("Label 1", red),
    #                        Label("Label 2", green),
    #                        Label("Label 3", blue)])
    model = LabelListModel()

    l = QVBoxLayout()
    w = QWidget(None)
    w.setLayout(l)
    addButton = QPushButton("Add random label\n note: \n the first added is permanent")
    l.addWidget(addButton)

    def addRandomLabel():
        model.insertRow(
            model.rowCount(),
            Label(
                "Label {}".format(model.rowCount() + 1),
                QColor(numpy.random.randint(0, 255), numpy.random.randint(0, 255), numpy.random.randint(0, 255)),
            ),
        )

    addButton.clicked.connect(addRandomLabel)

    model.makeRowPermanent(0)

    w.show()
    w.raise_()

    tableView = LabelListView()
    tableView.support_merges = True
    l.addWidget(tableView)
    tableView.setModel(model)

    tableView2 = LabelListView()
    tableView2.setModel(model)
    tableView2._table.setShowGrid(True)
    l.addWidget(tableView2)

    sys.exit(app.exec_())
