###############################################################################
#   volumina: volume slicing and editing library
#
#       Copyright (C) 2011-2014, the ilastik developers
#                                <team@ilastik.org>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the Lesser GNU General Public License
# as published by the Free Software Foundation; either version 2.1
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Lesser General Public License for more details.
#
# See the files LICENSE.lgpl2 and LICENSE.lgpl3 for full text of the
# GNU Lesser General Public License version 2.1 and 3 respectively.
# This information is also available on the ilastik web site at:
# 		   http://ilastik.org/license/
###############################################################################
"""Input and output from and to other libraries resp. formats.

Volumine works with 5d array-like objects assuming the coordinate
system (time, x, y, z, channel). This module provides methods to convert other
data types to this expected format.
"""
import numpy as np
from volumina.slicingtools import slicing2shape


class Array5d(object):
    """Embed a array with dim = 3 into the volumina coordinate system."""

    def __init__(self, array, dtype=np.uint8):
        assert len(array.shape) == 3
        self.a = array
        self.dtype = dtype

    def __getitem__(self, slicing):
        sl3d = (slicing[1], slicing[2], slicing[3])
        ret = np.zeros(slicing2shape(slicing), dtype=self.dtype)
        ret[0, :, :, :, 0] = self.a[tuple(sl3d)]
        return ret

    @property
    def shape(self):
        return (1,) + self.a.shape + (1,)

    def astype(self, dtype):
        return Array5d(self.a, dtype)
